<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Exports\SuperAdmin\TherapistEducationsExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\TherapistExperience\CreateRequest;
use App\Http\Requests\ImportRequest;
use App\Imports\SuperAdmin\TherapistEducationsImport;
use App\Models\TherapistExperience;
use App\Models\Therapist;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;

class TherapistExperiencesController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        $this->middleware('permission:therapist.add_experience');
        $this->middleware('permission:therapist.add_experience', ['only' => ['store']]);
        $this->middleware('permission:therapist.add_experience', ['only' => ['update']]);
        $this->middleware('permission:therapist.add_experience', ['only' => ['destroy']]);
        $this->middleware('permission:therapist.add_experience', ['only' => ['export']]);
        $this->middleware('permission:therapist.add_experience', ['only' => ['import']]);
    }
    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null, $therapist)
    {
        if ($req != null) {
            $therapist_experiences =  $therapist->therapist_experiences();
            if ($req->trash && $req->trash == 'with') {
                $therapist_experiences =  $therapist_experiences->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapist_experiences =  $therapist_experiences->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapist_experiences = $therapist_experiences->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapist_experiences = $therapist_experiences->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort_field != null && $req->sort_type != null) {
                $therapist_experiences = $therapist_experiences->OrderBy($req->sort_field, $req->sort_type);
            } else {
                $therapist_experiences = $therapist_experiences->OrderBy('id', 'desc');
            }
            if ($export != null) { // for export do not paginate
                $therapist_experiences = $therapist_experiences->get();
                return $therapist_experiences;
            }
            $therapist_experiences = $therapist_experiences->get();
            return $therapist_experiences;
        }
        $therapist_experiences = $therapist->therapist_experiences()->withAll()->orderBy('id', 'desc')->get();
        return $therapist_experiences;
    }


    /*********View All TherapistExperience  ***********/
    public function index(Request $request, Therapist $therapist)
    {
        $therapist_experiences = $this->getter($request, null, $therapist);
        return view('super_admins.therapists.therapist_experiences.index', compact('therapist_experiences', 'therapist'));
    }

    /*********View Create Form of TherapistExperience  ***********/
    public function create(Therapist $therapist)
    {
        return view('super_admins.therapists.therapist_experiences.create', compact('therapist'));
    }

    /*********Store TherapistExperience  ***********/
    public function store(CreateRequest $request, Therapist $therapist)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $data = $request->all();
            $data['image'] = uploadFile($request, 'file', 'therapist_experiences');
            $therapist_experience = $therapist->therapist_experiences()->create($data);
            $therapist_experience = $therapist->therapist_experiences()->withAll()->find($therapist_experience->id);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_experiences.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_experiences.index', $therapist->id)->with('message', 'Experience Created Successfully')->with('message_type', 'success');
    }

    /*********View TherapistExperience  ***********/
    public function show(Therapist $therapist, TherapistExperience $therapist_experience)
    {
        if ($therapist->id != $therapist_experience->therapist_id) {
            return redirect()->back()->with('message', 'TherapistExperience Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapists.therapist_experiences.show', compact('therapist_experience', 'therapist'));
    }

    /*********View Edit Form of TherapistExperience  ***********/
    public function edit(Therapist $therapist, TherapistExperience $therapist_experience)
    {
        if ($therapist->id != $therapist_experience->therapist_id) {
            return redirect()->back()->with('message', 'TherapistExperience Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapists.therapist_experiences.edit', compact('therapist_experience', 'therapist'));
    }

    /*********Update TherapistExperience  ***********/
    public function update(CreateRequest $request, Therapist $therapist, TherapistExperience $therapist_experience)
    {
        if ($therapist->id != $therapist_experience->therapist_id) {
            return redirect()->back()->with('message', 'TherapistExperience Not Found')->with('message_type', 'error');
        }
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $data = $request->all();
            if ($request->file) {
                $data['image'] = uploadFile($request, 'file', 'therapist_experiences', $therapist_experience->image);
            } else {
                $data['image'] = $therapist_experience->image;
            }
            $therapist_experience->update($data);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_experiences.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_experiences.index', $therapist->id)->with('message', 'TherapistExperience Updated Successfully')->with('message_type', 'success');
    }

    /********* Export  CSV And Excel  **********/
    public function export(Request $request)
    {
        $therapist_experiences = $this->getter($request, "export");
        if (in_array($request->export, ['csv,xlsx'])) {
            $extension = $request->export;
        } else {
            $extension = 'xlsx';
        }
        $filename = "therapist_experiences." . $extension;
        return Excel::download(new TherapistEducationsExport($therapist_experiences), $filename);
    }
    /********* Import CSV And Excel  **********/
    public function import(ImportRequest $request)
    {
        $file = $request->file('file');
        Excel::import(new TherapistEducationsImport, $file);
        return redirect()->back()->with('message', 'TherapistExperience Categories imported Successfully')->with('message_type', 'success');
    }


    /*********Soft DELETE TherapistExperience ***********/
    public function destroy(Therapist $therapist, TherapistExperience $therapist_experience)
    {
        if ($therapist->id != $therapist_experience->therapist_id) {
            return redirect()->back()->with('message', 'TherapistExperience Not Found')->with('message_type', 'error');
        }
        $therapist_experience->delete();
        return redirect()->back()->with('message', 'TherapistExperience Deleted Successfully')->with('message_type', 'success');
    }

    /*********Permanently DELETE TherapistExperience ***********/
    public function destroyPermanently(Request $request, Therapist $therapist, $therapist_experience)
    {
        $therapist_experience = TherapistExperience::withTrashed()->find($therapist_experience);
        if ($therapist_experience) {
            if ($therapist_experience->trashed()) {
                if ($therapist_experience->image && file_exists(public_path($therapist_experience->image))) {
                    unlink(public_path($therapist_experience->image));
                }
                $therapist_experience->forceDelete();
                return redirect()->back()->with('message', 'TherapistExperience Deleted Successfully')->with('message_type', 'success');
            } else {
                return redirect()->back()->with('message', 'TherapistExperience is Not in Trash')->with('message_type', 'error');
            }
        } else {
            return redirect()->back()->with('message', 'TherapistExperience Not Found')->with('message_type', 'error');
        }
    }
    /********* Restore TherapistExperience***********/
    public function restore(Request $request, Therapist $therapist, $therapist_experience)
    {
        $therapist_experience = TherapistExperience::withTrashed()->find($therapist_experience);
        if ($therapist_experience->trashed()) {
            $therapist_experience->restore();
            return redirect()->back()->with('message', 'TherapistExperience Restored Successfully')->with('message_type', 'success');
        } else {
            return redirect()->back()->with('message', 'TherapistExperience Not Found')->with('message_type', 'error');
        }
    }
}
